
'
'  DisplayStyleInfo.c
'  Copyright (c) 2002 by J Brown
'  Freeware
'
'
'  void FillStyleLists(HWND hwndTarget, HWND hwndStyleList, HWND hwndExStyleList,
'         BOOL fAllStyles, BOOL fExtControl)
'
'  Fill the two list-boxes (hwndStyleList and hwndExStyleList)
'  with the appropriate styles for the specified target window.
'
'  hwndTarget      - window to find styles for
'  hwndStyleList   - listbox to receive standard styles
'  hwndExStyleList - listbox to receive extended styles
'
'  fAllStyles      - FALSE - just adds the styles that are set
'                    TRUE  - adds ALL possible styles, but
'                            only selects those that are set
'
'  fExtControl     - include control-specific extended styles
'                    (i.e. LVS_EX_xxx styles, not present in the extended window styles
'
'
' v1.6.1 - fixed small bug thanks to Holger Stenger
'
#INCLUDE <tchar.h>

TYPE RetStyle
   style    AS DWORD    ' Single window style
   name     AS LPCTSTR  ' Textual name of style
   IsSet    AS BOOL
   ExtMsg   AS DWORD
END TYPE


TYPE StyleLookupEx
   style    AS DWORD    ' Single window style
   name     AS LPCTSTR  ' Textual name of style
   cmp_mask AS DWORD    ' If zero, then -style- is treated as a single bit-field
   ' Otherwise, cmp_mask is used to make sure that
   ' ALL the bits in the mask match -style-
   depends  AS DWORD    ' Style is only valid if includes these styles
   excludes AS DWORD    ' Style is only valid if these aren't set
END TYPE


'  Use this helper-macro to fill in the first two members
'  of the style structures.
'
'  e.g. STYLE_(WS_CHILD)  ->  WS_CHILD, "WS_CHILD"
'
CONST STYLE_(style) = (UINT)style, _T(#style)
CONST pSTYLE_(style) = style, UBOUND(style)


'  Use this structure to list each window class with its
'  associated style table.

TYPE ClassStyleLookup
   szClassName AS LPCTSTR
   stylelist   AS StyleLookupEx  PTR
   NumStyles
   dwData      AS DWORD
END TYPE


SET WindowStyles[] AS StyleLookupEx

   STYLE_(WS_OVERLAPPEDWINDOW), 0             , -1, (WS_POPUP|WS_CHILD),
   STYLE_(WS_POPUPWINDOW)     , WS_POPUPWINDOW, -1, 0                  ,
   STYLE_(WS_OVERLAPPED)      , 0             , -1, (WS_POPUP|WS_CHILD), '0x00000000
   STYLE_(WS_POPUP)       , 0, -1, 0, '0x80000000
   STYLE_(WS_CHILD)       , 0, -1, 0, '0x40000000
   STYLE_(WS_MINIMIZE)    , 0, -1, 0, '0x20000000
   STYLE_(WS_VISIBLE)     , 0, -1, 0, '0x10000000
   STYLE_(WS_DISABLED)    , 0, -1, 0, '0x08000000
   STYLE_(WS_CLIPSIBLINGS), 0, -1, 0, '0x04000000
   STYLE_(WS_CLIPCHILDREN), 0, -1, 0, '0x02000000
   STYLE_(WS_MAXIMIZE)    , 0, -1, 0, '0x01000000
   STYLE_(WS_CAPTION)     , 0, -1, 0, '0x00C00000

   '(BORDER|CAPTION)
   STYLE_(WS_DLGFRAME)  , 0, -1, 0, '0x00400000
   STYLE_(WS_BORDER)    , 0, -1, 0, '0x00800000
   STYLE_(WS_VSCROLL)   , 0, -1, 0, '0x00200000
   STYLE_(WS_HSCROLL)   , 0, -1, 0, '0x00100000
   STYLE_(WS_SYSMENU)   , 0, -1, 0, '0x00080000
   STYLE_(WS_THICKFRAME), 0, -1, 0, '0x00040000
   STYLE_(WS_GROUP)     , 0, -1, 0, '0x00020000
   STYLE_(WS_TABSTOP)   , 0, -1, 0, '0x00010000

'   STYLE_(WS_MINIMIZEBOX),  0, WS_POPUPWINDOW|WS_OVERLAPPEDWINDOW|WS_CAPTION, 0, '0x00020000
'   STYLE_(WS_MAXIMIZEBOX),  0, WS_POPUPWINDOW|WS_OVERLAPPEDWINDOW|WS_CAPTION, 0  '0x00010000
   STYLE_(WS_MINIMIZEBOX),  0, WS_SYSMENU, 0, '0x00020000
   STYLE_(WS_MAXIMIZEBOX),  0, WS_SYSMENU, 0  '0x00010000

END SET




'  Dialog box styles (class = #32770)
SET DialogStyles[] AS StyleLookupEx

   STYLE_(DS_ABSALIGN),     0, -1, 0, ' 0x00000001
   STYLE_(DS_SYSMODAL),     0, -1, 0, ' 0x00000002
   STYLE_(DS_LOCALEDIT),    0, -1, 0, ' 0x00000020
   STYLE_(DS_SETFONT),      0, -1, 0, ' 0x00000040
   STYLE_(DS_MODALFRAME),   0, -1, 0, ' 0x00000080
   STYLE_(DS_NOIDLEMSG),    0, -1, 0, ' 0x00000100
   STYLE_(DS_SETFOREGROUND),0, -1, 0, ' 0x00000200

   ' #if(WINVER >= 0x0400)

   STYLE_(DS_3DLOOK)      , 0, -1, 0, ' 0x00000004
   STYLE_(DS_FIXEDSYS)    , 0, -1, 0, ' 0x00000008
   STYLE_(DS_NOFAILCREATE), 0, -1, 0, ' 0x00000010
   STYLE_(DS_CONTROL)     , 0, -1, 0, ' 0x00000400
   STYLE_(DS_CENTER)      , 0, -1, 0, ' 0x00000800
   STYLE_(DS_CENTERMOUSE) , 0, -1, 0, ' 0x00001000
   STYLE_(DS_CONTEXTHELP) , 0, -1, 0  ' 0x00002000

END SET

'  Button styles (Button)
SET ButtonStyles[] AS StyleLookupEx

   STYLE_(BS_PUSHBUTTON),   0,   -1, BS_DEFPUSHBUTTON|BS_CHECKBOX|BS_AUTOCHECKBOX|BS_RADIOBUTTON|BS_GROUPBOX|BS_AUTORADIOBUTTON,
   STYLE_(BS_DEFPUSHBUTTON)  , 0xf, -1, 0, ' 0x0001
   STYLE_(BS_CHECKBOX)       , 0xf, -1, 0, ' 0x0002
   STYLE_(BS_AUTOCHECKBOX)   , 0xf, -1, 0, ' 0x0003
   STYLE_(BS_RADIOBUTTON)    , 0xf, -1, 0, ' 0x0004
   STYLE_(BS_3STATE)         , 0xf, -1, 0, ' 0x0005
   STYLE_(BS_AUTO3STATE)     , 0xf, -1, 0, ' 0x0006
   STYLE_(BS_GROUPBOX)       , 0xf, -1, 0, ' 0x0007
   STYLE_(BS_USERBUTTON)     , 0xf, -1, 0, ' 0x0008
   STYLE_(BS_AUTORADIOBUTTON), 0xf, -1, 0, ' 0x0009
   STYLE_(BS_OWNERDRAW)      , 0xf, -1, 0, ' 0x000B
   STYLE_(BS_LEFTTEXT)       , 0  , -1, 0, ' 0x0020

   ' winver >= 4.0 (index 42)
   STYLE_(BS_TEXT)       , 0, -1, (BS_ICON|BS_BITMAP|BS_AUTOCHECKBOX|BS_AUTORADIOBUTTON|BS_CHECKBOX|BS_RADIOBUTTON),' 0x00000000
   STYLE_(BS_ICON)       , 0, -1, 0, ' 0x0040
   STYLE_(BS_BITMAP)     , 0, -1, 0, ' 0x0080
   STYLE_(BS_LEFT)       , 0, -1, 0, ' 0x0100
   STYLE_(BS_RIGHT)      , 0, -1, 0, ' 0x0200
   STYLE_(BS_CENTER)     , 0, -1, 0, ' 0x0300
   STYLE_(BS_TOP)        , 0, -1, 0, ' 0x0400
   STYLE_(BS_BOTTOM)     , 0, -1, 0, ' 0x0800
   STYLE_(BS_VCENTER)    , 0, -1, 0, ' 0x0C00
   STYLE_(BS_PUSHLIKE)   , 0, -1, 0, ' 0x1000
   STYLE_(BS_MULTILINE)  , 0, -1, 0, ' 0x2000
   STYLE_(BS_NOTIFY)     , 0, -1, 0, ' 0x4000
   STYLE_(BS_FLAT)       , 0, -1, 0, ' 0x8000
   STYLE_(BS_RIGHTBUTTON), 0, -1, 0  ' BS_LEFTTEXT

END SET

'  Edit styles (Edit)
SET EditStyles[] AS StyleLookupEx

   STYLE_(ES_LEFT)       , 0, -1, (ES_CENTER|ES_RIGHT), ' 0x0000
   STYLE_(ES_CENTER)     , 0, -1, 0, ' 0x0001
   STYLE_(ES_RIGHT)      , 0, -1, 0, ' 0x0002
   STYLE_(ES_MULTILINE)  , 0, -1, 0, ' 0x0004
   STYLE_(ES_UPPERCASE)  , 0, -1, 0, ' 0x0008
   STYLE_(ES_LOWERCASE)  , 0, -1, 0, ' 0x0010
   STYLE_(ES_PASSWORD)   , 0, -1, 0, ' 0x0020
   STYLE_(ES_AUTOVSCROLL), 0, -1, 0, ' 0x0040
   STYLE_(ES_AUTOHSCROLL), 0, -1, 0, ' 0x0080
   STYLE_(ES_NOHIDESEL)  , 0, -1, 0, ' 0x0100
   STYLE_(ES_OEMCONVERT) , 0, -1, 0, ' 0x0400
   STYLE_(ES_READONLY)   , 0, -1, 0, ' 0x0800
   STYLE_(ES_WANTRETURN) , 0, -1, 0, ' 0x1000
   STYLE_(ES_NUMBER)     , 0, -1, 0  ' 0x2000

END SET

SET RichedStyles[] AS StyleLookupEx

   '  Standard edit control styles
   STYLE_(ES_LEFT)         , 0, -1, (ES_CENTER|ES_RIGHT),  ' 0x0000
   STYLE_  (ES_CENTER)     , 0, -1, 0, ' 0x0001
   STYLE_  (ES_RIGHT)      , 0, -1, 0, ' 0x0002
   STYLE_  (ES_MULTILINE)  , 0, -1, 0, ' 0x0004
   ' STYLE_(ES_UPPERCASE)  , 0, -1, 0, ' 0x0008
   ' STYLE_(ES_LOWERCASE)  , 0, -1, 0, ' 0x0010
   STYLE_  (ES_PASSWORD)   , 0, -1, 0, ' 0x0020
   STYLE_  (ES_AUTOVSCROLL), 0, -1, 0, ' 0x0040
   STYLE_  (ES_AUTOHSCROLL), 0, -1, 0, ' 0x0080
   STYLE_  (ES_NOHIDESEL)  , 0, -1, 0, ' 0x0100
   ' STYLE_(ES_OEMCONVERT) , 0, -1, 0, ' 0x0400
   STYLE_  (ES_READONLY)   , 0, -1, 0, ' 0x0800
   STYLE_  (ES_WANTRETURN) , 0, -1, 0, ' 0x1000
   STYLE_  (ES_NUMBER)     , 0, -1, 0, ' 0x2000

   '  Addition Rich Edit control styles

   STYLE_(ES_SAVESEL)        , 0, -1, 0, ' 0x00008000
   STYLE_(ES_SUNKEN)         , 0, -1, 0, ' 0x00004000
   STYLE_(ES_DISABLENOSCROLL), 0, -1, 0, ' 0x00002000
   STYLE_(ES_SELECTIONBAR)   , 0, -1, 0, ' 0x01000000
   STYLE_(ES_NOOLEDRAGDROP)  , 0, -1, 0  ' 0x00000008

END SET

'  Combo box styles (combobox)
SET ComboStyles[] AS StyleLookupEx

   STYLE_(CBS_SIMPLE)           , 0x3, -1, 0, ' 0x0001
   STYLE_(CBS_DROPDOWN)         , 0x3, -1, 0, ' 0x0002
   STYLE_(CBS_DROPDOWNLIST)     , 0x3, -1, 0, ' 0x0003
   STYLE_(CBS_OWNERDRAWFIXED)   , 0  , -1, 0, ' 0x0010
   STYLE_(CBS_OWNERDRAWVARIABLE), 0  , -1, 0, ' 0x0020
   STYLE_(CBS_AUTOHSCROLL)      , 0  , -1, 0, ' 0x0040
   STYLE_(CBS_OEMCONVERT)       , 0  , -1, 0, ' 0x0080
   STYLE_(CBS_SORT)             , 0  , -1, 0, ' 0x0100
   STYLE_(CBS_HASSTRINGS)       , 0  , -1, 0, ' 0x0200
   STYLE_(CBS_NOINTEGRALHEIGHT) , 0  , -1, 0, ' 0x0400
   STYLE_(CBS_DISABLENOSCROLL)  , 0  , -1, 0, ' 0x0800

   ' #if(WINVER >= 0x0400)
   STYLE_(CBS_UPPERCASE),   0, -1, 0,   ' 0x1000
   STYLE_(CBS_LOWERCASE),   0, -1, 0    ' 0x2000
   ' #endif

END SET

'  Listbox styles (Listbox)
SET ListBoxStyles[] AS StyleLookupEx

   STYLE_            (LBS_NOTIFY)           , 0, -1, 0 , ' 0x0001
   STYLE_            (LBS_SORT)             , 0, -1, 0 , ' 0x0002
   STYLE_            (LBS_NOREDRAW)         , 0, -1, 0 , ' 0x0004
   STYLE_            (LBS_MULTIPLESEL)      , 0, -1, 0 , ' 0x0008
   ' *check me*STYLE_(LBS_OWNERDRAWFIXED    , ), 0 , -1, 0, ' 0x0010
   STYLE_            (LBS_OWNERDRAWVARIABLE), 0, -1, 0 , ' 0x0020
   STYLE_            (LBS_HASSTRINGS)       , 0, -1, 0 , ' 0x0040
   STYLE_            (LBS_USETABSTOPS)      , 0, -1, 0 , ' 0x0080
   STYLE_            (LBS_NOINTEGRALHEIGHT) , 0, -1, 0 , ' 0x0100
   STYLE_            (LBS_MULTICOLUMN)      , 0, -1, 0 , ' 0x0200
   STYLE_            (LBS_WANTKEYBOARDINPUT), 0, -1, 0 , ' 0x0400
   STYLE_            (LBS_EXTENDEDSEL)      , 0, -1, 0 , ' 0x0800
   STYLE_            (LBS_DISABLENOSCROLL)  , 0, -1, 0 , ' 0x1000
   STYLE_            (LBS_NODATA)           , 0, -1, 0 , ' 0x2000
   STYLE_            (LBS_NOSEL)            , 0, -1, 0   ' 0x4000

END SET

'  Scrollbar control styles (Scrollbar)
SET ScrollbarStyles[] AS StyleLookupEx

   STYLE_(SBS_TOPALIGN)   , 0, SBS_HORZ, 0, ' 0x0002
   STYLE_(SBS_LEFTALIGN)  , 0, SBS_VERT, 0, ' 0x0002
   STYLE_(SBS_BOTTOMALIGN), 0, SBS_HORZ|SBS_SIZEBOX|SBS_SIZEGRIP, 0,  ' 0x0004
   STYLE_(SBS_RIGHTALIGN) , 0, SBS_VERT|SBS_SIZEBOX|SBS_SIZEGRIP, 0,  ' 0x0004
   STYLE_(SBS_HORZ)       , 0, -1, SBS_VERT|SBS_SIZEBOX|SBS_SIZEGRIP, ' 0x0000
   STYLE_(SBS_VERT)       , 0, -1, SBS_SIZEBOX|SBS_SIZEGRIP         , ' 0x0001
   STYLE_(SBS_SIZEBOXTOPLEFTALIGN)    , 0, SBS_SIZEBOX|SBS_SIZEGRIP, 0,       ' 0x0002
   STYLE_(SBS_SIZEBOXBOTTOMRIGHTALIGN), 0, SBS_SIZEBOX|SBS_SIZEGRIP, 0,       ' 0x0004
   STYLE_(SBS_SIZEBOX),     0, -1, 0,                 ' 0x0008
   STYLE_(SBS_SIZEGRIP),    0, -1, 0                  ' 0x0010

END SET

'  Static control styles (Static)
SET StaticStyles[] AS StyleLookupEx

   STYLE_(SS_LEFT)          , 0x1f, -1, SS_CENTER|SS_RIGHT, ' 0x0000
   STYLE_(SS_CENTER)        , 0x1f, -1, 0, ' 0x0001
   STYLE_(SS_RIGHT)         , 0x1f, -1, 0, ' 0x0002
   STYLE_(SS_ICON)          , 0x1f, -1, 0, ' 0x0003
   STYLE_(SS_BLACKRECT)     , 0x1f, -1, 0, ' 0x0004
   STYLE_(SS_GRAYRECT)      , 0x1f, -1, 0, ' 0x0005
   STYLE_(SS_WHITERECT)     , 0x1f, -1, 0, ' 0x0006
   STYLE_(SS_BLACKFRAME)    , 0x1f, -1, 0, ' 0x0007
   STYLE_(SS_GRAYFRAME)     , 0x1f, -1, 0, ' 0x0008
   STYLE_(SS_WHITEFRAME)    , 0x1f, -1, 0, ' 0x0009
   STYLE_(SS_USERITEM)      , 0x1f, -1, 0, ' 0x000A
   STYLE_(SS_SIMPLE)        , 0x1f, -1, 0, ' 0x000B
   STYLE_(SS_LEFTNOWORDWRAP), 0x1f, -1, 0, ' 0x000C

   STYLE_(SS_OWNERDRAW)  , 0x1f, -1, 0, ' 0x000D
   STYLE_(SS_BITMAP)     , 0x1f, -1, 0, ' 0x000E
   STYLE_(SS_ENHMETAFILE), 0x1f, -1, 0, ' 0x000F
   STYLE_(SS_ETCHEDHORZ) , 0x1f, -1, 0, ' 0x0010
   STYLE_(SS_ETCHEDVERT) , 0x1f, -1, 0, ' 0x0011
   STYLE_(SS_ETCHEDFRAME), 0x1f, -1, 0, ' 0x0012
   STYLE_(SS_TYPEMASK)   , 0x1f, -1, 0, ' 0x001F
   STYLE_(SS_NOPREFIX)   , 0   , -1, 0, ' 0x0080

   STYLE_(SS_NOTIFY)       , 0, -1, 0, ' 0x0100
   STYLE_(SS_CENTERIMAGE)  , 0, -1, 0, ' 0x0200
   STYLE_(SS_RIGHTJUST)    , 0, -1, 0, ' 0x0400
   STYLE_(SS_REALSIZEIMAGE), 0, -1, 0, ' 0x0800
   STYLE_(SS_SUNKEN)       , 0, -1, 0, ' 0x1000
   STYLE_(SS_ENDELLIPSIS)  , 0, -1, 0, ' 0x4000
   STYLE_(SS_PATHELLIPSIS) , 0, -1, 0, ' 0x8000
   STYLE_(SS_WORDELLIPSIS) , 0, -1, 0, ' 0xC000
   STYLE_(SS_ELLIPSISMASK) , 0, -1, 0  ' 0xC000

END SET

'   Standard Common controls styles
SET CommCtrlList[] AS StyleLookupEx

   STYLE_(CCS_TOP)          , 0x3, -1, 0, ' 0x0001
   STYLE_(CCS_NOMOVEY)      , 0x3, -1, 0, ' 0x0002
   STYLE_(CCS_BOTTOM)       , 0x3, -1, 0, ' 0x0003
   STYLE_(CCS_NORESIZE)     , 0  , -1, 0, ' 0x0004
   STYLE_(CCS_NOPARENTALIGN), 0  , -1, 0, ' 0x0008

   STYLE_(CCS_ADJUSTABLE)   , 0, -1, 0, ' 0x0020
   STYLE_(CCS_NODIVIDER)    , 0, -1, 0, ' 0x0040

   ' #if (_WIN32_IE >= 0x0300)
   STYLE_(CCS_VERT)         , 0, -1, 0  ' 0x0080
   ' #endif

END SET

'   DragList - uses same styles as listview

'  Header control (SysHeader32)
SET HeaderStyles[] AS StyleLookupEx

   STYLE_(HDS_HORZ),      0, -1, 16,    ' 0x0000
   STYLE_(HDS_BUTTONS),     0, -1, 0,   ' 0x0002

   ' #if (_WIN32_IE >= 0x0300)
   STYLE_(HDS_HOTTRACK),    0, -1, 0,   ' 0x0004
   STYLE_(HDS_DRAGDROP),    0, -1, 0,   ' 0x0040
   STYLE_(HDS_FULLDRAG),    0, -1, 0,   ' 0x0080
   ' #endif

   STYLE_(HDS_HIDDEN),      0, -1, 0,   ' 0x0008

   ' #if (_WIN32_IE >= 0x0500)
   STYLE_(HDS_FILTERBAR),   0, -1, 0    ' 0x0100
   ' #endif

END SET

'  Listview (SysListView32)
SET ListViewStyles[] AS StyleLookupEx

   STYLE_(LVS_ICON),    LVS_TYPEMASK, -1, LVS_REPORT|LVS_SMALLICON|LVS_LIST, ' 0x0000
   STYLE_(LVS_REPORT)   , LVS_TYPEMASK, -1, 0, ' 0x0001
   STYLE_(LVS_SMALLICON), LVS_TYPEMASK, -1, 0, ' 0x0002
   STYLE_(LVS_LIST)     , LVS_TYPEMASK, -1, 0, ' 0x0003

   STYLE_(LVS_SINGLESEL)      , 0, -1, 0, ' 0x0004
   STYLE_(LVS_SHOWSELALWAYS)  , 0, -1, 0, ' 0x0008
   STYLE_(LVS_SORTASCENDING)  , 0, -1, 0, ' 0x0010
   STYLE_(LVS_SORTDESCENDING) , 0, -1, 0, ' 0x0020
   STYLE_(LVS_SHAREIMAGELISTS), 0, -1, 0, ' 0x0040
   STYLE_(LVS_NOLABELWRAP)    , 0, -1, 0, ' 0x0080
   STYLE_(LVS_AUTOARRANGE)    , 0, -1, 0, ' 0x0100
   STYLE_(LVS_EDITLABELS)     , 0, -1, 0, ' 0x0200

   ' #if (_WIN32_IE >= 0x0300)
   STYLE_(LVS_OWNERDATA), 0, -1, 0,   ' 0x1000
   ' #endif

   STYLE_(LVS_NOSCROLL),  0, -1, 0,   ' 0x2000

   STYLE_(LVS_ALIGNTOP),  0, -1, 0,   ' 0x0000
   STYLE_(LVS_ALIGNLEFT), LVS_ALIGNMASK, -1, 0, ' 0x0800

   ' STYLE_(LVS_ALIGNMASK), 0, -1, 0,   ' 0x0c00
   ' STYLE_(LVS_TYPESTYLEMASK), 0, -1, 0,   ' 0xfc00

   STYLE_(LVS_OWNERDRAWFIXED),  0, -1, 0,   ' 0x0400
   STYLE_(LVS_NOCOLUMNHEADER),  0, -1, 0,   ' 0x4000
   STYLE_(LVS_NOSORTHEADER)  ,  0, -1, 0    ' 0x8000

END SET

'  Toolbar control (ToolbarWindow32)
SET ToolbarStyles[] AS StyleLookupEx

   STYLE_(TBSTYLE_TOOLTIPS), 0, -1, 0,   ' 0x0100
   STYLE_(TBSTYLE_WRAPABLE), 0, -1, 0,   ' 0x0200
   STYLE_(TBSTYLE_ALTDRAG) , 0, -1, 0,   ' 0x0400

   ' #if (_WIN32_IE >= 0x0300)
   STYLE_(TBSTYLE_FLAT),  0, -1, 0,   ' 0x0800
   STYLE_(TBSTYLE_LIST),  0, -1, 0,   ' 0x1000
   STYLE_(TBSTYLE_CUSTOMERASE), 0, -1, 0,   ' 0x2000
   ' #endif

   ' #if (_WIN32_IE >= 0x0400)
   STYLE_(TBSTYLE_REGISTERDROP),0, -1, 0,   ' 0x4000
   STYLE_(TBSTYLE_TRANSPARENT), 0, -1, 0    ' 0x8000
   ' #endif

END SET

'  Rebar control (RebarControl32)
SET RebarStyles[] AS StyleLookupEx

   ' #if (_WIN32_IE >= 0x0400)
   STYLE_(RBS_TOOLTIPS)       , 0, -1, 0, ' 0x0100
   STYLE_(RBS_VARHEIGHT)      , 0, -1, 0, ' 0x0200
   STYLE_(RBS_BANDBORDERS)    , 0, -1, 0, ' 0x0400
   STYLE_(RBS_FIXEDORDER)     , 0, -1, 0, ' 0x0800
   STYLE_(RBS_REGISTERDROP)   , 0, -1, 0, ' 0x1000
   STYLE_(RBS_AUTOSIZE)       , 0, -1, 0, ' 0x2000
   STYLE_(RBS_VERTICALGRIPPER), 0, -1, 0, ' 0x4000
   STYLE_(RBS_DBLCLKTOGGLE)   , 0, -1, 0  ' 0x8000
   ' #endif

END SET

'  Track Bar control (msctls_trackbar32)
SET TrackbarStyles[] AS StyleLookupEx

   STYLE_(TBS_AUTOTICKS), 0xf, -1, 0       , ' 0x0001
   STYLE_(TBS_VERT)     , 0xf, -1, 0       , ' 0x0002
   STYLE_(TBS_HORZ)     , 0xf, -1, TBS_VERT, ' 0x0000
   STYLE_(TBS_TOP)      , 0xf, -1, 0       , ' 0x0004
   STYLE_(TBS_BOTTOM)   , 0xf, -1, TBS_TOP , ' 0x0000
   STYLE_(TBS_LEFT)     , 0xf, -1, 0       , ' 0x0004
   STYLE_(TBS_RIGHT)    , 0xf, -1, TBS_LEFT, ' 0x0000
   STYLE_(TBS_BOTH)     , 0xf, -1, 0       , ' 0x0008

   STYLE_(TBS_NOTICKS)       , 0, -1, 0, ' 0x0010
   STYLE_(TBS_ENABLESELRANGE), 0, -1, 0, ' 0x0020
   STYLE_(TBS_FIXEDLENGTH)   , 0, -1, 0, ' 0x0040
   STYLE_(TBS_NOTHUMB)       , 0, -1, 0, ' 0x0080

   ' #if (_WIN32_IE >= 0x0300)
   STYLE_(TBS_TOOLTIPS),    0, -1, 0,   ' 0x0100
   ' #endif

   ' #if (_WIN32_IE >= 0x0500)
   STYLE_(TBS_REVERSED),    0, -1, 0    ' 0x0200
   ' #endif

END SET

'  Treeview (SysTreeView32)
SET TreeViewStyles[] AS StyleLookupEx

   STYLE_(TVS_HASBUTTONS)     , 0, -1, 0, ' 0x0001
   STYLE_(TVS_HASLINES)       , 0, -1, 0, ' 0x0002
   STYLE_(TVS_LINESATROOT)    , 0, -1, 0, ' 0x0004
   STYLE_(TVS_EDITLABELS)     , 0, -1, 0, ' 0x0008
   STYLE_(TVS_DISABLEDRAGDROP), 0, -1, 0, ' 0x0010
   STYLE_(TVS_SHOWSELALWAYS)  , 0, -1, 0, ' 0x0020

   ' #if (_WIN32_IE >= 0x0300)
   STYLE_(TVS_RTLREADING),    0, -1, 0,   ' 0x0040
   STYLE_(TVS_NOTOOLTIPS),    0, -1, 0,   ' 0x0080
   STYLE_(TVS_CHECKBOXES),    0, -1, 0,   ' 0x0100
   STYLE_(TVS_TRACKSELECT),   0, -1, 0,   ' 0x0200

   ' #if (_WIN32_IE >= 0x0400)
   STYLE_(TVS_SINGLEEXPAND) , 0, -1, 0, ' 0x0400
   STYLE_(TVS_INFOTIP)      , 0, -1, 0, ' 0x0800
   STYLE_(TVS_FULLROWSELECT), 0, -1, 0, ' 0x1000
   STYLE_(TVS_NOSCROLL)     , 0, -1, 0, ' 0x2000
   STYLE_(TVS_NONEVENHEIGHT), 0, -1, 0, ' 0x4000

   ' #if (_WIN32_IE >= 0x500)
   STYLE_(TVS_NOHSCROLL),   0, -1, 0    ' 0x8000

   ' #endif
   ' #endif
   ' #endif

END SET

'  Tooltips (tooltips_class32)
SET ToolTipStyles[] AS StyleLookupEx

   STYLE_(TTS_ALWAYSTIP),   0, -1, 0,   ' 0x01
   STYLE_(TTS_NOPREFIX),    0, -1, 0,   ' 0x02

   ' #if (_WIN32_IE >= 0x0500)
   STYLE_(TTS_NOANIMATE),   0, -1, 0,   ' 0x10
   STYLE_(TTS_NOFADE),      0, -1, 0,   ' 0x20
   STYLE_(TTS_BALLOON),     0, -1, 0    ' 0x40
   ' #endif

END SET

'  Statusbar (msctls_statusbar32)
SET StatusBarStyles[] AS StyleLookupEx

   STYLE_(SBARS_SIZEGRIP),    0, -1, 0,   ' 0x0100

   ' #if (_WIN32_IE >= 0x0400)
   STYLE_(SBT_TOOLTIPS),    0, -1, 0    ' 0x0800
   ' #endif

END SET

'  Updown control
SET UpDownStyles[] AS StyleLookupEx

   STYLE_(UDS_WRAP)       , 0, -1, 0, ' 0x0001
   STYLE_(UDS_SETBUDDYINT), 0, -1, 0, ' 0x0002
   STYLE_(UDS_ALIGNRIGHT) , 0, -1, 0, ' 0x0004
   STYLE_(UDS_ALIGNLEFT)  , 0, -1, 0, ' 0x0008
   STYLE_(UDS_AUTOBUDDY)  , 0, -1, 0, ' 0x0010
   STYLE_(UDS_ARROWKEYS)  , 0, -1, 0, ' 0x0020
   STYLE_(UDS_HORZ)       , 0, -1, 0, ' 0x0040
   STYLE_(UDS_NOTHOUSANDS), 0, -1, 0, ' 0x0080

   ' #if (_WIN32_IE >= 0x0300)
   STYLE_(UDS_HOTTRACK),    0, -1, 0   ' 0x0100
   ' #endif

END SET

'  Progress control (msctls_progress32)
SET ProgressStyles[] AS StyleLookupEx

   ' #if (_WIN32_IE >= 0x0300)
   STYLE_(PBS_SMOOTH),      0, -1, 0,   ' 0x01
   STYLE_(PBS_VERTICAL),    0, -1, 0    ' 0x04
   ' #endif

END SET

'  Tab control (SysTabControl32)
SET TabStyles[] AS StyleLookupEx

   ' #if (_WIN32_IE >= 0x0300)
   STYLE_(TCS_SCROLLOPPOSITE),    0, -1, 0,   ' 0x0001   '  assumes multiline tab
   STYLE_(TCS_BOTTOM),      0, TCS_VERTICAL, 0, ' 0x0002
   STYLE_(TCS_RIGHT),     0, -1, 0,   ' 0x0002
   STYLE_(TCS_MULTISELECT),   0, -1, 0,   ' 0x0004
   ' #endif
   ' #if (_WIN32_IE >= 0x0400)
   STYLE_(TCS_FLATBUTTONS)   , 0, -1, 0, ' 0x0008
   ' #endif
   STYLE_(TCS_FORCEICONLEFT) , 0, -1, 0, ' 0x0010
   STYLE_(TCS_FORCELABELLEFT), 0, -1, 0, ' 0x0020
   ' #if (_WIN32_IE > = 0x0300)
   STYLE_(TCS_HOTTRACK)      , 0, -1, 0, ' 0x0040
   STYLE_(TCS_VERTICAL)      , 0, -1, 0, ' 0x0080
   ' #endif
   STYLE_(TCS_TABS)             , 0, -1, TCS_BUTTONS  , ' 0x0000
   STYLE_(TCS_BUTTONS)          , 0, -1, 0            , ' 0x0100
   STYLE_(TCS_SINGLELINE)       , 0, -1, TCS_MULTILINE, ' 0x0000
   STYLE_(TCS_MULTILINE)        , 0, -1, 0            , ' 0x0200
   STYLE_(TCS_RIGHTJUSTIFY)     , 0, -1, -1           , ' 0x0000
   STYLE_(TCS_FIXEDWIDTH)       , 0, -1, 0            , ' 0x0400
   STYLE_(TCS_RAGGEDRIGHT)      , 0, -1, 0            , ' 0x0800
   STYLE_(TCS_FOCUSONBUTTONDOWN), 0, -1, 0            , ' 0x1000
   STYLE_(TCS_OWNERDRAWFIXED)   , 0, -1, 0            , ' 0x2000
   STYLE_(TCS_TOOLTIPS)         , 0, -1, 0            , ' 0x4000
   STYLE_(TCS_FOCUSNEVER)       , 0, -1, 0              ' 0x8000

END SET

'  Animation control (SysAnimate32)
SET AnimateStyles[] AS StyleLookupEx

   STYLE_(ACS_CENTER)     , 0, -1, 0, ' 0x0001
   STYLE_(ACS_TRANSPARENT), 0, -1, 0, ' 0x0002
   STYLE_(ACS_AUTOPLAY)   , 0, -1, 0, ' 0x0004

   ' #if (_WIN32_IE >= 0x0300)
   STYLE_(ACS_TIMER),     0, -1, 0    ' 0x0008
   ' #endif

END SET

'  Month-calendar control (SysMonthCal32)
SET MonthCalStyles[] AS StyleLookupEx

   STYLE_(MCS_DAYSTATE)   , 0, -1, 0, ' 0x0001
   STYLE_(MCS_MULTISELECT), 0, -1, 0, ' 0x0002
   STYLE_(MCS_WEEKNUMBERS), 0, -1, 0, ' 0x0004

   ' #if (_WIN32_IE >= 0x0400)
   STYLE_(MCS_NOTODAYCIRCLE), 0, -1, 0,   ' 0x0008
   STYLE_(MCS_NOTODAY)      , 0, -1, 0   ' 0x0010
   ' #endif

END SET

'  Date-Time picker (SysDateTimePick32)
SET DateTimeStyles[] AS StyleLookupEx

   STYLE_(DTS_RIGHTALIGN),    0, -1, 0,   ' 0x0020
   STYLE_(DTS_APPCANPARSE),   0, -1, 0,   ' 0x0010
   STYLE_(DTS_SHORTDATECENTURYFORMAT),  0, -1, 0,   ' 0x000C   ' #if (_WIN32_IE >= 0x500)

   STYLE_(DTS_TIMEFORMAT),    0, -1, 0,   ' 0x0009
   STYLE_(DTS_LONGDATEFORMAT) , 0, -1, 0                 , ' 0x0004
   STYLE_(DTS_SHOWNONE)       , 0, -1, 0                 , ' 0x0002
   STYLE_(DTS_UPDOWN)         , 0, -1, 0                 , ' 0x0001
   STYLE_(DTS_SHORTDATEFORMAT), 0, -1, 0                   ' 0x0000

END SET

'  Pager control (SysPager)
SET PagerStyles[] AS StyleLookupEx

   ' Pager control
   STYLE_(PGS_VERT)      , 0, -1, PGS_HORZ, ' 0x0000
   STYLE_(PGS_HORZ)      , 0, -1, 0       , ' 0x0001
   STYLE_(PGS_AUTOSCROLL), 0, -1, 0       , ' 0x0002
   STYLE_(PGS_DRAGNDROP) , 0, -1, 0         ' 0x0004

END SET

'Window styles
CONST STYLE_NOSEL      = 0x01
CONST STYLE_NOFOCUS    = 0x02
CONST STYLE_HGRIDLINES = 0x04
CONST STYLE_VGRIDLINES = 0x08
CONST STYLE_GRIDFRAME  = 0x10
CONST STYLE_NOCOLSIZE  = 0x20

'  Grid control (RAGrid)
SET RAGridStyles[] AS StyleLookupEx

   ' Grid control
   STYLE_(STYLE_NOSEL)        , 0, -1, 0       , ' 0x0001
   STYLE_(STYLE_NOFOCUS)      , 0, -1, 0       , ' 0x0002
   STYLE_(STYLE_HGRIDLINES)   , 0, -1, 0       , ' 0x0004
   STYLE_(STYLE_VGRIDLINES)   , 0, -1, 0       , ' 0x0008
   STYLE_(STYLE_GRIDFRAME)    , 0, -1, 0       , ' 0x0010
   STYLE_(STYLE_NOCOLSIZE )   , 0, -1, 0         ' 0x0020

END SET


'  Extended window styles (for all windows)
SET StyleExList[] AS StyleLookupEx

   STYLE_(WS_EX_DLGMODALFRAME) , 0, -1, 0, ' 0x00000001L
   STYLE_(WS_EX_NOPARENTNOTIFY), 0, -1, 0, ' 0x00000004L
   STYLE_(WS_EX_TOPMOST)       , 0, -1, 0, ' 0x00000008L
   STYLE_(WS_EX_ACCEPTFILES)   , 0, -1, 0, ' 0x00000010L
   STYLE_(WS_EX_TRANSPARENT)   , 0, -1, 0, ' 0x00000020L

   ' #if(WINVER >= 0x0400)

   STYLE_(WS_EX_MDICHILD)  ,  0, -1, 0, ' 0x00000040L
   STYLE_(WS_EX_TOOLWINDOW),  0, -1, 0, ' 0x00000080L
   STYLE_(WS_EX_WINDOWEDGE),  0, -1, 0, ' 0x00000100L
   STYLE_(WS_EX_CLIENTEDGE),  0, -1, 0, ' 0x00000200L
   STYLE_(WS_EX_CONTEXTHELP), 0, -1, 0, ' 0x00000400L

   STYLE_(WS_EX_LEFT)          , 0, -1, (WS_EX_RIGHT)        , ' 0x00000000L
   STYLE_(WS_EX_RIGHT)         , 0, -1, 0                    , ' 0x00001000L
   STYLE_(WS_EX_LTRREADING)    , 0, -1, (WS_EX_RTLREADING)   , ' 0x00000000L
   STYLE_(WS_EX_RTLREADING)    , 0, -1, 0                    , ' 0x00002000L
   STYLE_(WS_EX_LEFTSCROLLBAR) , 0, -1, 0                    , ' 0x00004000L
   STYLE_(WS_EX_RIGHTSCROLLBAR), 0, -1, (WS_EX_LEFTSCROLLBAR), ' 0x00000000L

   STYLE_(WS_EX_CONTROLPARENT), 0, -1, 0, ' 0x00010000L
   STYLE_(WS_EX_STATICEDGE)   , 0, -1, 0, ' 0x00020000L
   STYLE_(WS_EX_APPWINDOW)    , 0, -1, 0, ' 0x00040000L

   STYLE_(WS_EX_OVERLAPPEDWINDOW),  0, -1, 0 ' (WS_EX_WINDOWEDGE | WS_EX_CLIENTEDGE)
   ' STYLE_(WS_EX_PALETTEWINDOW), 0, -1, 0, ' (WS_EX_WINDOWEDGE | WS_EX_TOOLWINDOW | WS_EX_TOPMOST)

   ' #endif

END SET

'  ListView extended styles
SET ListViewExStyles[] AS StyleLookupEx

   ' ListView control styles
   STYLE_(LVS_EX_GRIDLINES)       , 0, -1, 0, ' 0x00000001
   STYLE_(LVS_EX_SUBITEMIMAGES)   , 0, -1, 0, ' 0x00000002
   STYLE_(LVS_EX_CHECKBOXES)      , 0, -1, 0, ' 0x00000004
   STYLE_(LVS_EX_TRACKSELECT)     , 0, -1, 0, ' 0x00000008
   STYLE_(LVS_EX_HEADERDRAGDROP)  , 0, -1, 0, ' 0x00000010
   STYLE_(LVS_EX_FULLROWSELECT)   , 0, -1, 0, ' 0x00000020
   STYLE_(LVS_EX_ONECLICKACTIVATE), 0, -1, 0, ' 0x00000040
   STYLE_(LVS_EX_TWOCLICKACTIVATE), 0, -1, 0, ' 0x00000080
   ' #if (_WIN32_IE >= 0x0400)
   STYLE_(LVS_EX_FLATSB)        , 0, -1, 0, ' 0x00000100
   STYLE_(LVS_EX_REGIONAL)      , 0, -1, 0, ' 0x00000200
   STYLE_(LVS_EX_INFOTIP)       , 0, -1, 0, ' 0x00000400
   STYLE_(LVS_EX_UNDERLINEHOT)  , 0, -1, 0, ' 0x00000800
   STYLE_(LVS_EX_UNDERLINECOLD) , 0, -1, 0, ' 0x00001000
   STYLE_(LVS_EX_MULTIWORKAREAS), 0, -1, 0, ' 0x00002000
   ' #endif
   ' #if (_WIN32_IE >= 0x0500)
   STYLE_(LVS_EX_LABELTIP),   0, -1, 0      ' 0x00004000
   ' #endif

END SET

'  ComboBoxEx extended styles
SET ComboBoxExStyles[] AS StyleLookupEx

   STYLE_(CBES_EX_NOEDITIMAGE)      , 0, -1, 0, ' 0x00000001
   STYLE_(CBES_EX_NOEDITIMAGEINDENT), 0, -1, 0, ' 0x00000002
   STYLE_(CBES_EX_PATHWORDBREAKPROC), 0, -1, 0, ' 0x00000004

   ' #if(_WIN32_IE >= 0x0400)
   STYLE_(CBES_EX_NOSIZELIMIT),   0, -1, 0, ' 0x00000008
   STYLE_(CBES_EX_CASESENSITIVE), 0, -1, 0  ' 0x00000010
   ' #endif

END SET

'  Tab control extended styles
SET TabCtrlExStyles[] AS StyleLookupEx

   STYLE_(TCS_EX_FLATSEPARATORS), 0, -1, 0, ' 0x00000001
   STYLE_(TCS_EX_REGISTERDROP),   0, -1, 0  ' 0x00000002
END SET

'  Toolbar extended styles
SET ToolBarExStyles[] AS StyleLookupEx

   ' #if (_WIN32_IE >= 0x0400)
   STYLE_(TBSTYLE_EX_DRAWDDARROWS)      , 0, -1, 0, ' 0x0001

   ' #if (_WIN32_IE >= 0x0501)
   STYLE_(TBSTYLE_EX_MIXEDBUTTONS)      , 0, -1, 0, ' 0x0008
   STYLE_(TBSTYLE_EX_HIDECLIPPEDBUTTONS), 0, -1, 0  ' 0x0010

   ' #endif
   ' #endif

END SET

'  Support RichEdit Event masks!!!
SET RichedEventMask[] AS StyleLookupEx

   STYLE_(ENM_NONE)         , 0, -1, -1, ' 0x00000000
   STYLE_(ENM_CHANGE)       , 0, -1, 0 , ' 0x00000001
   STYLE_(ENM_UPDATE)       , 0, -1, 0 , ' 0x00000002
   STYLE_(ENM_SCROLL)       , 0, -1, 0 , ' 0x00000004
   STYLE_(ENM_KEYEVENTS)    , 0, -1, 0 , ' 0x00010000
   STYLE_(ENM_MOUSEEVENTS)  , 0, -1, 0 , ' 0x00020000
   STYLE_(ENM_REQUESTRESIZE), 0, -1, 0 , ' 0x00040000
   STYLE_(ENM_SELCHANGE)    , 0, -1, 0 , ' 0x00080000
   STYLE_(ENM_DROPFILES)    , 0, -1, 0 , ' 0x00100000
   STYLE_(ENM_PROTECTED)    , 0, -1, 0 , ' 0x00200000
   STYLE_(ENM_CORRECTTEXT)  , 0, -1, 0 , ' 0x00400000 ' PenWin specific
   STYLE_(ENM_SCROLLEVENTS) , 0, -1, 0 , ' 0x00000008
   STYLE_(ENM_DRAGDROPDONE) , 0, -1, 0 , ' 0x00000010

   '  Far East specific notification mask
   STYLE_(ENM_IMECHANGE)      , 0, -1, 0, ' 0x00800000 ' unused by RE2.0
   STYLE_(ENM_LANGCHANGE)     , 0, -1, 0, ' 0x01000000
   STYLE_(ENM_OBJECTPOSITIONS), 0, -1, 0, ' 0x02000000
   STYLE_(ENM_LINK)           , 0, -1, 0  ' 0x04000000

END SET

'
'   Lookup table which matches window classnames to style-lists
'
SET StandardControls[] AS ClassStyleLookup

   _T("#32770")            , pSTYLE_(DialogStyles)  , 0,
   _T("Button")            , pSTYLE_(ButtonStyles)   , 0,
   _T("ComboBox")          , pSTYLE_(ComboStyles)    , 0,
   _T("Edit")              , pSTYLE_(EditStyles)     , 0,
   _T("ListBox")           , pSTYLE_(ListBoxStyles)  , 0,
   _T("RICHEDIT")          , pSTYLE_(RichedStyles)   , 0,
   _T("RichEdit20A")       , pSTYLE_(RichedStyles)   , 0,
   _T("RichEdit20W")       , pSTYLE_(RichedStyles)   , 0,
   _T("Scrollbar")         , pSTYLE_(ScrollbarStyles), 0,
   _T("Static")            , pSTYLE_(StaticStyles)   , 0,
   _T("SysAnimate32")      , pSTYLE_(AnimateStyles)  , 0,
   _T("ComboBoxEx")        , pSTYLE_(ComboStyles)    , 0, ' (Just a normal combobox)
   _T("SysDateTimePick32") , pSTYLE_(DateTimeStyles) , 0,
   _T("DragList")          , pSTYLE_(ListBoxStyles)  , 0, ' (Just a normal list)
   _T("SysHeader32")       , pSTYLE_(HeaderStyles)   , 0,
   '"SysIPAddress32"       , pSTYLE_(IPAddressStyles), 0,   (NO STYLES)
   _T("SysListView32")     , pSTYLE_(ListViewStyles) , 0,
   _T("SysMonthCal32")     , pSTYLE_(MonthCalStyles) , 0,
   _T("SysPager")          , pSTYLE_(PagerStyles)    , 0,
   _T("msctls_progress32") , pSTYLE_(ProgressStyles) , 0,
   _T("RebarWindow32")     , pSTYLE_(RebarStyles)    , 0,
   _T("msctls_statusbar32"), pSTYLE_(StatusBarStyles), 0,
   '"SysLink"              , pSTYLE_(SysLinkStyles)  , 0,   (DO IT!)
   _T("SysTabControl32")   , pSTYLE_(TabStyles)      , 0,
   _T("ToolbarWindow32")   , pSTYLE_(ToolbarStyles)  , 0,
   _T("tooltips_class32")  , pSTYLE_(ToolTipStyles)  , 0,
   _T("msctls_trackbar32") , pSTYLE_(TrackbarStyles) , 0,
   _T("SysTreeView32")     , pSTYLE_(TreeViewStyles) , 0,
   _T("msctls_updown32")   , pSTYLE_(UpDownStyles)   , 0,
   _T("RAGrid")            , pSTYLE_(RAGridStyles)   , 0,
   _T(""), 0, 0, 0
END SET

' Classes which use the CCS_xxx styles
SET CustomControls[] AS ClassStyleLookup

   _T("msctls_statusbar32"), pSTYLE_(CommCtrlList), 0,
   _T("RebarWindow32")     , pSTYLE_(CommCtrlList), 0,
   _T("ToolbarWindow32")   , pSTYLE_(CommCtrlList), 0,
   _T("SysHeader32")       , pSTYLE_(CommCtrlList), 0,
   _T(""),0, 0, 0
END SET



' Classes which have extended window styles
SET ExtendedControls[] AS ClassStyleLookup

   _T("SysTabControl32"),  pSTYLE_(TabCtrlExStyles),    TCM_GETEXTENDEDSTYLE,
   _T("ToolbarWindow32"),  pSTYLE_(ToolBarExStyles),    TB_GETEXTENDEDSTYLE,
   _T("ComboBox"),         pSTYLE_(ComboBoxExStyles),   CBEM_GETEXTENDEDSTYLE,
   _T("SysListView32"),    pSTYLE_(ListViewExStyles),   LVM_GETEXTENDEDLISTVIEWSTYLE,
   _T("RICHEDIT"),         pSTYLE_(RichedEventMask),    EM_GETEVENTMASK,
   _T("RichEdit20A"),      pSTYLE_(RichedEventMask),    EM_GETEVENTMASK,
   _T("RichEdit20W"),      pSTYLE_(RichedEventMask),    EM_GETEVENTMASK,
   _T(""),0,  0,  0
END SET


'
' Match the window classname to a pClassList
' - a lookup table of classname / matching stylelist
'
'Return index to ClassStyleLookup
'
FUNCTION FindStyleList(pClassList AS ClassStyleLookup PTR, szClassName AS TCHAR PTR, pdwData AS DWORD PTR)
   DIM RAW i=0

   WHILE pClassList[i].stylelist <> 0
      IF stricmp(szClassName, pClassList[i].szClassName) = 0 THEN
         IF pdwData THEN *pdwData = pClassList[i].dwData
         FUNCTION = i 'pClassList[i].stylelist
      END IF
      INCR i
   WEND

   FUNCTION = -1
END FUNCTION


'  Find all the styles that match from the specified list
'
'  StyleList  - list of styles
'  hwndList   - listbox to add styles to
'  dwStyle    - style for the target window
'  fAllStyles -
'
FUNCTION EnumStyles(MasterStyle$, NumStyles, StyleList AS StyleLookupEx PTR, dwStyle ,fAllStyles AS BOOL ) AS DWORD

   ' Remember what the style is before we start modifying it
   DIM RAW dwOrig = dwStyle AS DWORD
   DIM RAW i=0
   DIM RAW fAddIt AS BOOL
   DIM RAW pStyle AS StyleLookupEx PTR

   local dynamic SL[NumStyles+1] as INTEGER
   
   QSORTIDX SL, NumStyles+1, StyleList.style, 3

   ' Loop through all of the styles that we know about.
   ' Check each style against our window's one, to see if it is set or not.

   'WHILE StyleList[i].style <> -1
   FOR i = NumStyles TO 0 STEP -1

      fAddIt = FALSE
      pStyle = &StyleList[SL[i]]

      ' This style needs a mask to detect if it is set -
      ' i.e. the style doesn't just use one bit to decide if it is on/off.
      IF pStyle->cmp_mask <> 0 THEN

         'if((StyleList[i].depends & dwStyle) != dwStyle)
         '  continue;

         ' Style is valid if the excludes styles are not set
         IF (pStyle->excludes <> 0) AND (pStyle->excludes BAND (dwOrig BAND pStyle->cmp_mask) = 0) THEN
            fAddIt = TRUE
         END IF
         ' If the style matches exactly (when masked)
         IF (pStyle->style <> 0) AND ((pStyle->cmp_mask BAND dwStyle) = pStyle->style) THEN
            fAddIt = TRUE
         END IF
      ELSE
         ' Style is valid when
         IF (pStyle->excludes <> 0) AND (pStyle->excludes BAND dwOrig) = 0 THEN
            fAddIt = TRUE
         END IF

         ' If style matches exactly (all bits are set)
         IF (pStyle->style <> 0) AND (pStyle->style BAND dwStyle) = pStyle->style THEN
            fAddIt = TRUE
            ' If no bits are set, then we have to skip it
         ELSEIF (pStyle->style <> 0) AND (pStyle->style BAND dwStyle) = 0 THEN
            fAddIt = FALSE
         END IF

         ' If this style depends on others being set..
         IF dwStyle AND (pStyle->depends BAND dwStyle) = 0 THEN
            fAddIt = FALSE
         END IF
      END IF

      ' Now add the style..
      IF fAddIt = TRUE OR fAllStyles THEN

         ' We've added this style, so remove it to stop it appearing again
         IF fAddIt THEN dwStyle &= ~(pStyle->style)

         ' Add to list, and set the list's extra item data to the style's value

         IF pStyle->style <> 0 THEN
            IF MasterStyle[0] THEN CONCAT(MasterStyle, "|")
            CONCAT(MasterStyle, pStyle->name)
         END IF

      END IF
      'INCR i
   NEXT i

   ' return the style. This will be zero if we decoded all the bits
   ' that were set, or non-zero if there are still bits left

   FUNCTION = dwStyle
END FUNCTION



'  Find all the styles that match from the specified list
'
'  StyleList  - list of styles
'  hwndList   - listbox to add styles to
'  dwStyle    - style for the target window
'  fAllStyles -
'
FUNCTION EnumStylesToArray(StyPropArray AS RetStyle PTR, BYREF StyCnt, NumStyles, StyleList AS StyleLookupEx PTR, dwStyle) AS DWORD

   ' Remember what the style is before we start modifying it
   DIM RAW dwOrig = dwStyle AS DWORD
   DIM RAW i=0
   DIM RAW fAddIt AS BOOL
   DIM RAW pStyle AS StyleLookupEx PTR

   ' Loop through all of the styles that we know about.
   ' Check each style against our window's one, to see if it is set or not.
   local dynamic SL[NumStyles+1] as INTEGER
   
   QSORTIDX SL, NumStyles+1, StyleList.style, 3

   'WHILE StyleList[i].style <> -1
   FOR i = NumStyles TO 0 STEP -1

      fAddIt = FALSE
      pStyle = &StyleList[SL[i]]

      ' This style needs a mask to detect if it is set -
      ' i.e. the style doesn't just use one bit to decide if it is on/off.
      IF pStyle->cmp_mask <> 0 THEN

         'if((StyleList[i].depends & dwStyle) != dwStyle)
         '  continue;

         ' Style is valid if the excludes styles are not set
         IF (pStyle->excludes <> 0) AND (pStyle->excludes BAND (dwOrig BAND pStyle->cmp_mask) = 0) THEN
            fAddIt = TRUE
         END IF
         ' If the style matches exactly (when masked)
         IF (pStyle->style <> 0) AND ((pStyle->cmp_mask BAND dwStyle) = pStyle->style) THEN
            fAddIt = TRUE
         END IF
      ELSE
         ' Style is valid when
         IF (pStyle->excludes <> 0) AND (pStyle->excludes BAND dwOrig) = 0 THEN
            fAddIt = TRUE
         END IF

         ' If style matches exactly (all bits are set)
         IF (pStyle->style <> 0) AND (pStyle->style BAND dwStyle) = pStyle->style THEN
            fAddIt = TRUE
            ' If no bits are set, then we have to skip it
         ELSEIF (pStyle->style <> 0) AND (pStyle->style BAND dwStyle) = 0 THEN
            fAddIt = FALSE
         END IF

         ' If this style depends on others being set..
         IF dwStyle AND (pStyle->depends BAND dwStyle) = 0 THEN
            fAddIt = FALSE
         END IF
      END IF

      ' Now add the style..
      ' We've added this style, so remove it to stop it appearing again
      IF fAddIt THEN dwStyle &= ~ (pStyle->style)

      ' Add to list, and set the list's extra item data to the style's value
      IF StyPropArray AND pStyle->style <> 0 THEN
         StyPropArray[StyCnt].style = pStyle->style
         StyPropArray[StyCnt].name  = pStyle->name
         StyPropArray[StyCnt].IsSet = fAddIt
         INCR StyCnt
      END IF

      'INCR i
   NEXT i

   ' return the style. This will be zero if we decoded all the bits
   ' that were set, or non-zero if there are still bits left
   FUNCTION = dwStyle
END FUNCTION




'  This function takes HWNDs of two ListBoxes, which we will fill
'  with the style strings that are set for the specified window
'
'  Either hwndStyleList or hwndExStyleList can be NULL

FUNCTION FillStyleLists$(hwndTarget AS HWND ,hwndStyleList ,hwndExStyleList ,fAllStyles AS BOOL )

   DIM AUTO MasterStyle$
   DIM RAW szClassName[256] AS CHAR
   DIM RAW dwStyle AS DWORD
   DIM RAW dwStyleEx AS DWORD
   DIM RAW dwMessage AS DWORD

   DIM RAW ClassStyleIdx

   'window class
   GetClassName(hwndTarget, szClassName, SIZEOF(szClassName))

   'normal window styles
   dwStyle = GetWindowLong(hwndTarget, GWL_STYLE) BAND (~WS_DISABLED)

   IF hwndStyleList THEN
      ' enumerate the standard window styles, for any window no
      ' matter what class it might be.
      dwStyle = EnumStyles( MasterStyle$, UBOUND(WindowStyles), WindowStyles, dwStyle, fAllStyles)

      ' if the window class is one we know about, then see if we
      ' can decode any more style bits.
      ' enumerate the custom control styles
      ClassStyleIdx = FindStyleList(StandardControls, szClassName, 0)
      IF ClassStyleIdx <> -1 THEN
         dwStyle = EnumStyles( MasterStyle$, StandardControls[ClassStyleIdx].NumStyles, StandardControls[ClassStyleIdx].stylelist, dwStyle, fAllStyles)
      END IF

      ' does the window support the CCS_xxx styles (custom control styles)
      ClassStyleIdx = FindStyleList(CustomControls, szClassName, 0)
      IF ClassStyleIdx <> -1 THEN
         dwStyle = EnumStyles( MasterStyle$, CustomControls[ClassStyleIdx].NumStyles, CustomControls[ClassStyleIdx].stylelist, dwStyle, fAllStyles)
      END IF

      ' if there are still style bits set in the window style,
      ' then there is something that we can't decode.
      ' Just display a single HEX entry at the end of the list.
      IF dwStyle <> 0 THEN
         IF MasterStyle[0] THEN CONCAT(MasterStyle, "|")
         MasterStyle$ = MasterStyle$ + "0x" + HEX$(dwStyle)
      END IF
   END IF


   ' Extended window styles
   IF hwndExStyleList THEN
      ' Find extended styles
      dwStyleEx = GetWindowLong(hwndTarget, GWL_EXSTYLE)
      EnumStyles(MasterStyle$, UBOUND(StyleExList), StyleExList, dwStyleEx, fAllStyles)

      ' Does this window use any custom control extended styles???
      ' If it does, then dwMessage will contain the message identifier to send
      ' to the window to retrieve them
      ClassStyleIdx = FindStyleList(ExtendedControls, szClassName, &dwMessage)
      ' Add them if required
      IF ClassStyleIdx <> -1 THEN
         dwStyleEx = SendMessage(hwndTarget, dwMessage, 0, 0)
         EnumStyles( MasterStyle$, ExtendedControls[ClassStyleIdx].NumStyles, ExtendedControls[ClassStyleIdx].stylelist, dwStyleEx, fAllStyles)
      END IF
   END IF

   FUNCTION = MasterStyle$
END FUNCTION



'  This function takes HWNDs of two ListBoxes, which we will fill
'  with the style strings that are set for the specified window
'
'  Either hwndStyleList or hwndExStyleList can be NULL

FUNCTION FillStyleArray(StyPropArray AS RetStyle PTR, hwndTarget AS HWND ,hwndStyleList ,hwndExStyleList)

   DIM RAW szClassName[256] AS CHAR
   DIM RAW dwStyle   AS DWORD
   DIM RAW dwStyleEx AS DWORD
   DIM RAW dwMessage AS DWORD
   DIM RAW StyCnt = 0

   DIM RAW ClassStyleIdx

   'window class
   GetClassName(hwndTarget, szClassName, SIZEOF(szClassName))

   'normal window styles
   dwStyle = GetWindowLong(hwndTarget, GWL_STYLE)

   IF hwndStyleList = 1  THEN
      ' enumerate the standard window styles, for any window no
      ' matter what class it might be.
      dwStyle = EnumStylesToArray(StyPropArray, &StyCnt, UBOUND(WindowStyles), WindowStyles, dwStyle)

   ELSEIF hwndStyleList = 2 THEN
      dwStyle = EnumStylesToArray(0, &StyCnt, UBOUND(WindowStyles), WindowStyles, dwStyle)

      ' if the window class is one we know about, then see if we
      ' can decode any more style bits.
      ' enumerate the custom control styles
      ClassStyleIdx = FindStyleList(StandardControls, szClassName, 0)
      IF ClassStyleIdx <> -1 THEN
         dwStyle = EnumStylesToArray(StyPropArray, &StyCnt, StandardControls[ClassStyleIdx].NumStyles, StandardControls[ClassStyleIdx].stylelist, dwStyle)
      END IF

      ' does the window support the CCS_xxx styles (custom control styles)
      ClassStyleIdx = FindStyleList(CustomControls, szClassName, 0)
      IF ClassStyleIdx <> -1 THEN
         dwStyle = EnumStylesToArray(StyPropArray, &StyCnt, CustomControls[ClassStyleIdx].NumStyles, CustomControls[ClassStyleIdx].stylelist, dwStyle)
      END IF

      ' if there are still style bits set in the window style,
      ' then there is something that we can't decode.
      ' Just display a single HEX entry at the end of the list.
      IF StyPropArray AND dwStyle <> 0 THEN
         StyPropArray[StyCnt].style = dwStyle
         StyPropArray[StyCnt].name  = ("0x" + HEX$(dwStyle))
         StyPropArray[StyCnt].IsSet = TRUE
      END IF
   END IF


   ' Extended window styles
   IF hwndExStyleList = 1 THEN
      ' Find extended styles
      dwStyleEx = GetWindowLong(hwndTarget, GWL_EXSTYLE)
      EnumStylesToArray(StyPropArray, &StyCnt, UBOUND(StyleExList), StyleExList, dwStyleEx)

      ' Does this window use any custom control extended styles???
      ' If it does, then dwMessage will contain the message identifier to send
      ' to the window to retrieve them
   ELSEIF hwndExStyleList = 2 THEN
      ClassStyleIdx = FindStyleList(ExtendedControls, szClassName, &dwMessage)
      ' Add them if required
      IF ClassStyleIdx <> -1 THEN
         dwStyleEx = SendMessage(hwndTarget, dwMessage, 0, 0)
         IF EnumStylesToArray(StyPropArray, &StyCnt, ExtendedControls[ClassStyleIdx].NumStyles, ExtendedControls[ClassStyleIdx].stylelist, dwStyleEx) THEN
            StyPropArray[0].ExtMsg = dwMessage
         END IF
      END IF
   END IF

   FUNCTION = StyCnt
END FUNCTION

